#include <stdio.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <syscall_thunk.h>
#include <errno.h>
#include <sys/socket.h>
#include <linux/net.h>
#include <linux/un.h>
#include <linux/in.h>

extern FILE * x86_logfile;
extern int x86_trace_current_syscall;

static struct param_transtbl sol_nametab[] = {
	{ X86_SO_DEBUG, SO_DEBUG },
	{ X86_SO_REUSEADDR, SO_REUSEADDR },
	{ X86_SO_TYPE, SO_TYPE },
	{ X86_SO_ERROR, SO_ERROR },
	{ X86_SO_DONTROUTE, SO_DONTROUTE },
	{ X86_SO_BROADCAST, SO_BROADCAST },
	{ X86_SO_SNDBUF, SO_SNDBUF },
	{ X86_SO_RCVBUF, SO_RCVBUF },
	{ X86_SO_KEEPALIVE, SO_KEEPALIVE },
	{ X86_SO_OOBINLINE, SO_OOBINLINE },
	{ X86_SO_NO_CHECK, SO_NO_CHECK },
	{ X86_SO_PRIORITY, SO_PRIORITY },
	{ X86_SO_LINGER, SO_LINGER },
	{ X86_SO_BSDCOMPAT, SO_BSDCOMPAT }
};

#define SOL_NAMETAB_SIZE sizeof(sol_nametab)/sizeof(struct param_transtbl)

char * get_domain_name(int d)
{
    switch(d) {
	case AF_UNIX:	return("AF_UNIX");
	case AF_INET:	return("AF_INET");
	default:	return("Unknown");
    }
}

char * get_type_name(int t)
{
    switch(t) {
	case SOCK_STREAM:	return("SOCK_STREAM");
	case SOCK_DGRAM:	return("SOCK_DGRAM");
	case SOCK_RAW:	return("SOCK_RAW");
	case SOCK_SEQPACKET:	return("SOCK_SEQPACKET");
	case SOCK_RDM:	return("SOCK_RDM");
	default:	return("Unknown");
    }
}

void print_sockaddr(struct sockaddr * sap)
{
    switch(sap->sa_family) {
	case AF_UNSPEC:
	    fprintf(x86_logfile, "[UNSPEC]");
	    break;
	case AF_AX25:
	    fprintf(x86_logfile, "[AX25]");
	    break;
	case AF_IPX:
	    fprintf(x86_logfile, "[IPX]");
	    break;
	case AF_APPLETALK:
	    fprintf(x86_logfile, "[APPLETALK]");
	    break;
	case AF_NETROM:
	    fprintf(x86_logfile, "[NETROM]");
	    break;
	case AF_BRIDGE:
	    fprintf(x86_logfile, "[BRIDGE]");
	    break;
	case AF_AAL5:
	    fprintf(x86_logfile, "[AAL5]");
	    break;
	case AF_X25:
	    fprintf(x86_logfile, "[X25]");
	    break;
	case AF_INET6:
	    fprintf(x86_logfile, "[INET6]");
	    break;
	case AF_UNIX: {
	    struct sockaddr_un * sunp = (struct sockaddr_un *)sap;
	    fprintf(x86_logfile, "[UNIX: %s]", sunp->sun_path);
	    break;
	}
	case AF_INET: {
	    struct sockaddr_in * sinp = (struct sockaddr_in *)sap;
	    fprintf(x86_logfile, "[INET: %d.%d.%d.%d / %d]",
			sinp->sin_addr.s_addr & 0xff,
			(sinp->sin_addr.s_addr >> 8) & 0xff,
			(sinp->sin_addr.s_addr >> 16) & 0xff,
			(sinp->sin_addr.s_addr >> 24) & 0xff,
			sinp->sin_port);
	    break;
	}
	default:
	    fprintf(x86_logfile, "[Unknown]");
	    break;
    }
}

char * get_socket_optname(int o)
{
    switch(o) {
	case SO_DEBUG:	return("SO_DEBUG");
	case SO_REUSEADDR:	return("SO_REUSEADDR");
	case SO_TYPE:	return("SO_TYPE");
	case SO_ERROR:	return("SO_ERROR");
	case SO_DONTROUTE:	return("SO_DONTROUTE");
	case SO_BROADCAST:	return("SO_BROADCAST");
	case SO_SNDBUF:	return("SO_SNDBUF");
	case SO_RCVBUF:	return("SO_RCVBUF");
	case SO_KEEPALIVE:	return("SO_KEEPALIVE");
	case SO_OOBINLINE:	return("SO_OOBINLINE");
	case SO_NO_CHECK:	return("SO_NO_CHECK");
	case SO_PRIORITY:	return("SO_PRIORITY");
	case SO_LINGER:	return("SO_LINGER");
	case SO_BSDCOMPAT:	return("SO_BSDCOMPAT");
	default:		return("Unknown");
    }
}

char * get_tcp_optname(int o)
{
    switch(o) {
	case IP_TOS:	return("IP_TOS");
	case IP_TTL:	return("IP_TTL");
	case IP_HDRINCL:	return("IP_HDRINCL");
	case IP_OPTIONS:	return("IP_OPTIONS");
	case IP_MULTICAST_IF:	return("IP_MULTICAST_IF");
	case IP_MULTICAST_TTL:	return("IP_MULTICAST_TTL");
	case IP_MULTICAST_LOOP:	return("IP_MULTICAST_LOOP");
	case IP_ADD_MEMBERSHIP:	return("IP_ADD_MEMBERSHIP");
	case IP_DROP_MEMBERSHIP:	return("IP_DROP_MEMBERSHIP");
	default:		return("Unknown");
    }
}

/* This is a dispatch point for seventeen socket-related system calls... */

int do_socketcall(int a0, int a1, int a2, int a3, int a4, int a5)
{

    int		retval;
    unsigned int *argp = ARG_TO_POINTER(a1);

    switch(a0) {
	case SYS_SOCKET: {
	    if(x86_trace_current_syscall) {
		fprintf(x86_logfile, "=> socket(%d(%s), %d(%s), %d)\n", 
			argp[0], get_domain_name(argp[0]),
			argp[1], get_type_name(argp[1]),
			argp[2]);
		fflush(x86_logfile);
	    }

	    retval = socket(argp[0], argp[1], argp[2]);
	    break;
	}
	case SYS_BIND: {
	    struct sockaddr *myaddr = ARG_TO_POINTER(argp[1]);
	    if(x86_trace_current_syscall) {
		fprintf(x86_logfile, "=> bind(%d, 0x%x", argp[0], myaddr);
		print_sockaddr(myaddr);
		fprintf(x86_logfile, ", %d)\n", argp[2]);
		fflush(x86_logfile);
	    }
	    retval = bind(argp[0], myaddr, argp[2]);
	    break;
	}
	case SYS_CONNECT: {
	    struct sockaddr *servaddr = ARG_TO_POINTER(argp[1]);
	    if(x86_trace_current_syscall) {
		fprintf(x86_logfile, "=> connect(%d, 0x%x", argp[0], servaddr);
		print_sockaddr(servaddr);
		fprintf(x86_logfile, ", %d)\n", argp[2]);
		fflush(x86_logfile);
	    }
	    retval = connect(argp[0], servaddr, argp[2]);
	    break;
	}
	case SYS_LISTEN: {
	    if(x86_trace_current_syscall) {
		fprintf(x86_logfile, "=> listen(%d, %d)\n", argp[0], argp[1]);
		fflush(x86_logfile);
	    }
	    retval = listen(argp[0], argp[1]);
	    break;
	}
	case SYS_ACCEPT: {
	    struct sockaddr *addr = ARG_TO_POINTER(argp[1]);
	    size_t *addrlen = ARG_TO_POINTER(argp[2]);
	    if(x86_trace_current_syscall) {
		fprintf(x86_logfile, "=> accept(%d, 0x%x", argp[0], addr);
		print_sockaddr(addr);
		fprintf(x86_logfile, ", 0x%x(=> %d)\n", addrlen, *addrlen);
		fflush(x86_logfile);
	    }
	    retval = accept(argp[0], addr, addrlen);
	    break;
	}
	case SYS_GETSOCKNAME: {
	    struct sockaddr *name = ARG_TO_POINTER(argp[1]);
	    size_t *namelen = ARG_TO_POINTER(argp[2]);
	    if(x86_trace_current_syscall) {
		fprintf(x86_logfile, "=> getsockname(%d, 0x%x", argp[0], name);
		print_sockaddr(name);
		fprintf(x86_logfile, ", 0x%x(=> %d)\n", namelen, *namelen);
		fflush(x86_logfile);
	    }
	    retval = getsockname(argp[0], name, namelen);
	    break;
	}
	case SYS_GETPEERNAME: {
	    struct sockaddr *name = ARG_TO_POINTER(argp[1]);
	    size_t *namelen = ARG_TO_POINTER(argp[2]);
	    if(x86_trace_current_syscall) {
		fprintf(x86_logfile, "=> getpeername(%d, 0x%x", argp[0], name);
		print_sockaddr(name);
		fprintf(x86_logfile, ", 0x%x(=> %d)\n", namelen, *namelen);
		fflush(x86_logfile);
	    }
	    retval = getpeername(argp[0], name, namelen);
	    break;
	}
	case SYS_SOCKETPAIR: {
	    int *sv = ARG_TO_POINTER(argp[3]);
	    if(x86_trace_current_syscall) {
		fprintf(x86_logfile, "=> socketpair(%d, %d[%s], %d[%s], 0x%x)\n",
			argp[0], argp[1], get_type_name(argp[1]), 
			argp[2], get_domain_name(argp[2]), sv);
		fflush(x86_logfile);
	    }
	    retval = socketpair(argp[0], argp[1], argp[2], sv);
	    if(x86_trace_current_syscall) {
		fprintf(x86_logfile, "     sockets returned on fd's %d & %d\n", sv[0], sv[1]);
		fflush(x86_logfile);
	    }
	    break;
	}	
	case SYS_SEND: {
	    void* msgp = ARG_TO_POINTER(argp[1]);
	    if(x86_trace_current_syscall) {
		fprintf(x86_logfile, "=> send(%d, 0x%x, %d, 0x%x)\n", 
				argp[0], msgp, argp[2], argp[3]);
		fflush(x86_logfile);
	    }
	    retval = send(argp[0], msgp, argp[2], argp[3]);
	    break;
	}
	case SYS_RECV:{
	    void* msgp = ARG_TO_POINTER(argp[1]);
	    if(x86_trace_current_syscall) {
		fprintf(x86_logfile, "=> recv(%d, 0x%x, %d, 0x%x)\n", 
				argp[0], msgp, argp[2], argp[3]);
		fflush(x86_logfile);
	    }
	    retval = recv(argp[0], msgp, argp[2], argp[3]);
	    break;
	}
	case SYS_SENDTO: {
	    void* msgp = ARG_TO_POINTER(argp[1]);
	    struct sockaddr *to = ARG_TO_POINTER(argp[4]);
	    if(x86_trace_current_syscall) {
		fprintf(x86_logfile, "=> sendto(%d, 0x%x, %d, 0x%x, 0x%x",
			argp[0], msgp, argp[2], argp[3], to);
		print_sockaddr(to);
		fprintf(x86_logfile, ", %d\n", argp[5]);
		fflush(x86_logfile);
	    }
	    retval = sendto(argp[0], msgp, argp[2], argp[3], to, argp[5]);
	    break;
	}
	case SYS_RECVFROM: {
	    void* msgp = ARG_TO_POINTER(argp[1]);
	    struct sockaddr *from = ARG_TO_POINTER(argp[4]);
	    size_t *fromlen = ARG_TO_POINTER(argp[5]);
	    if(x86_trace_current_syscall) {
		fprintf(x86_logfile, "=> recvfrom(%d, 0x%x, %d, 0x%x, 0x%x",
			argp[0], msgp, argp[2], argp[3], from);
		print_sockaddr(from);
		fprintf(x86_logfile, ", 0x%x(%d))\n", fromlen, *fromlen);
		fflush(x86_logfile);
	    }
	    retval = recvfrom(argp[0], msgp, argp[2], argp[3], from, fromlen);
	    break;
	}
	case SYS_SHUTDOWN: {
	    if(x86_trace_current_syscall) {
		fprintf(x86_logfile, "=> shutdown(0x%x, 0x%x)\n", a0, a1);
		fflush(x86_logfile);
	    }
	    retval = shutdown(a0, a1);
	}
	case SYS_SETSOCKOPT: {
	    int		sock=argp[0];
	    int		level=argp[1];
	    int		optname;
	    int		optlen = argp[4];

	    switch(level) {
		case SOL_TCP:
		    /* Exactly the same across both platforms... */
	    	    optname=argp[2];
		    if(x86_trace_current_syscall) {
			fprintf(x86_logfile, "=> setsockopt(%d, %d(SOL_TCP), %d(%s), 0x%x, %d)\n",
				sock, level, optname, get_tcp_optname(optname), argp[3], optlen);
			fflush(x86_logfile);
		    }
		    retval=setsockopt(sock, level, optname, 
					ARG_TO_POINTER(argp[3]), optlen);
		    break;
		case SOL_SOCKET:
	    	    optname=x86_to_alpha_param(argp[2], sol_nametab,
						SOL_NAMETAB_SIZE);

		    if(x86_trace_current_syscall) {
			fprintf(x86_logfile, "=> setsockopt(%d, %d(SOL_SOCKET), %d(%s), 0x%x, %d)\n",
				sock, level, optname, get_socket_optname(optname), argp[3], optlen);
			fflush(x86_logfile);
		    }
		    retval=setsockopt(sock, level, optname, 
					ARG_TO_POINTER(argp[3]), optlen);
		    break;
		default:
		    /* Can't deal with this... */
		    if(x86_trace_current_syscall) {
			fprintf(x86_logfile, "=> setsockopt(%d, %d(Unknown), %d, 0x%x, %d)\n",
				argp[0], argp[1], argp[2], argp[3], argp[4]);
			fflush(x86_logfile);
		    }
		    fprintf(x86_logfile, "Unknown sockopt level %d\n", level);
		    retval = -1;
		    errno = EINVAL;
		    break;
	    }
	    break;
	}

	case SYS_GETSOCKOPT: {
	    int		sock=argp[0];
	    int		level=argp[1];
	    int		optname;
	    size_t	*optlen = ARG_TO_POINTER(argp[4]);

	    switch(level) {
		case SOL_TCP:
		    /* Exactly the same across both platforms... */
		    optname = argp[2];
		    if(x86_trace_current_syscall) {
			fprintf(x86_logfile, "=> getsockopt(%d, %d(SOL_TCP), %d(%s), 0x%x, %d)\n",
				sock, level, optname, get_tcp_optname(optname), argp[3], optlen);
			fflush(x86_logfile);
		    }
		    retval=getsockopt(sock, level, optname, 
					ARG_TO_POINTER(argp[3]), optlen);
		    break;
		case SOL_SOCKET:
		    /* Fortunately, socket setsockopt's seem to be bit-for-bit
		     * compatible on both Intel and Alpha, except for the
		     * option names...
		     */
	    	    optname=x86_to_alpha_param(argp[2], sol_nametab,
						SOL_NAMETAB_SIZE);
		    if(x86_trace_current_syscall) {
			fprintf(x86_logfile, "=> setsockopt(%d, %d(SOL_SOCKET), %d(%s), 0x%x, %d)\n",
				sock, level, optname, get_socket_optname(optname), argp[3], optlen);
			fflush(x86_logfile);
		    }
		    retval=getsockopt(sock, level, optname, 
					ARG_TO_POINTER(argp[3]), optlen);
		    break;
		default:
		    /* Can't deal with this... */
		    if(x86_trace_current_syscall) {
			fprintf(x86_logfile, "=> setsockopt(%d, %d(Unknown), %d, 0x%x, %d)\n",
				argp[0], argp[1], argp[2], argp[3], argp[4]);
			fflush(x86_logfile);
		    }
		    fprintf(x86_logfile, "Unknown sockopt level %d\n", level);
		    retval = -1;
		    errno = EINVAL;
		    break;
	    }
	    break;
	}

	case SYS_RECVMSG:
	case SYS_SENDMSG: {
	    int sock = argp[0];
	    struct x86_msghdr *	msgp = ARG_TO_POINTER(argp[1]);
	    struct msghdr alpha_msg;
	    int flags = argp[2];
	    struct x86_iovec * xiop = ARG_TO_POINTER(msgp->msg_iov);
	    struct iovec	alpha_iovec[X86_UIO_MAXIOV];
	    int		i;

	    if(x86_trace_current_syscall) {
		fprintf(x86_logfile, "=> %s(%d, 0x%x, 0x%x)\n",
			(a0 == SYS_SENDMSG) ? "sendmsg" : "recvmsg",
			sock, msgp, flags);
		fflush(x86_logfile);
	    }

	    alpha_msg.msg_name = ARG_TO_POINTER(msgp->msg_name);
	    alpha_msg.msg_namelen = msgp->msg_namelen;
	    alpha_msg.msg_iov = alpha_iovec;
	    alpha_msg.msg_iovlen = msgp->msg_iovlen;
	    alpha_msg.msg_control = msgp->msg_control;
	    alpha_msg.msg_controllen = msgp->msg_controllen;
	    alpha_msg.msg_flags = msgp->msg_flags;

	    for(i = 0; i < msgp->msg_iovlen; i++) {
		alpha_iovec[i].iov_base = ARG_TO_POINTER(xiop[i].iov_base);
		alpha_iovec[i].iov_len = xiop[i].iov_len;
	    }

	    if(a0 == SYS_SENDMSG) {
		retval = sendmsg(sock, &alpha_msg, flags);
	    }
	    else {
		retval = recvmsg(sock, &alpha_msg, flags);
	    }
	}

	default:
	    return(-alpha_to_intel_errno(EINVAL));
    }

    if(retval == -1) {
	return(-alpha_to_intel_errno(errno));
    }
    else {
	return(retval);
    }
}
